//////////////////////////////////////////////////////////////////////////////////
//                                                                              //
// Amyuni RTPDF-32 - PDF Library for On Time RTOS-32                            //
//                                                                              //
// Copyright AMYUNI Lda. - AMYUNI Technologies                                  //
// 1998-2021, All Rights Reserved                   ***** CONFIDENTIAL *****    //
//                                                                              //
// Permission to use this work for any purpose must be expressly obtained       //
// from: AMYUNI Technologies, https://www.amyuni.com, management@amyuni.com     //
//                                                                              //
//////////////////////////////////////////////////////////////////////////////////
//
// acTable.h: interface for the acTable class.
// The acTable class is the representation of the RTPDF-32 table class
//
//////////////////////////////////////////////////////////////////////

#if !defined(ACTABLE_H_INCLUDED_)
#define ACTABLE_H_INCLUDED_

// forward declarations
class acReportRow;
class acReportCell;

class acTable
{
/** @name Table Class
* \anchor TableClass
*  
* @code
	//
	// This sample code shows how to create a table object, set the size of rows and columns and set the attributes and contents of cells
	//
	acTable	table;
	RECT	rc = { FLOAT2LONG(50.0f), FLOAT2LONG(50.0f), FLOAT2LONG(580.0f), FLOAT2LONG(290.0f) };
	int		rows = 5, cols = 4;
	acReportCell *cell;

	// intialize the table to 5 rows by 4 columns
	table.InitTable(rows, cols, rc);

	table.SetColWidth(0, FLOAT2LONG(150.0f), TRUE);		// enlarge first column
	table.SetColWidth(2, FLOAT2LONG(175.0f), FALSE);	// enlarge column 3 and reduce column 4	
	table.SetRowHeight(0, FLOAT2LONG(25.0f), 1);		// header row a bit smaller

	// create a default font for all the table
	HANDLE	hFnt1 = RTPdfGetFont(pdev, "Helvetica", 0, 0, 0, 1, 0);
	// create another font for the cell titles
	HANDLE  hFnt2 = RTPdfGetFont(pdev, "Helvetica-Bold", 0, 0, 0, 1, 0);

	// initialize all cell attributes
	for (int r = 0; r < rows; r++)
		for (int c = 0; c < cols; c++)
		{
			acReportCell *cell = table.GetCell(r, c);

			cell->horzBorders = acReportCell::HorzBorders::acHorzBorderBoth;
			cell->font = hFnt1;
			cell->pointSize = 11.5f;
			cell->titleFont = hFnt2;
			cell->titlePointSize = 9.0f;
			cell->borderWidth = 10;

			// make the first row different from the others
			if (r == 0)
			{
				cell->backColor = RGB(0, 0xAA, 0xFF);
				cell->textColor = RGB(0xFF, 0xFF, 0xFF);
			}
			else
			{
				cell->vertBorders = acReportCell::VertBorders::acVertBorderBoth;

				if (c == cols - 1)
				{
					// text in last column is right-aligned
					cell->horzAlign = acReportCell::HorzAlign::Right;
				}
			}
		}

	// merge the cells of the first row and set some text
	table.MergeCells(0, 0, cols - 1, 0);
	cell = table.GetCell(0, 0);
	cell->horzAlign = acReportCell::HorzAlign::HCentered;
	cell->SetText("Header Row");

	// merge 4 cells and set a background image and a title
	table.MergeCells(1, 2, 2, 3);
	cell = table.GetCell(2, 1);
	cell->horzAlign = acReportCell::HorzAlign::HCentered;
	cell->backBmp = GetImageFromFile(pdev, "c:\\temp\\logo.png");
	cell->SetTitleText("Image with a title");

	// cell with title and long text string clipped to boundaries
	cell = table.GetCell(3, 0);
	cell->SetTitleText("Title");
	cell->SetText("Cell contents clipped to boundaries");
	cell->clipToBoundaries = TRUE;

	// right-aligned text
	cell = table.GetCell(2, cols - 1);
	cell->SetText("0.00");
	
	table.Draw(pdev);

* @endcode
*/
///@{

public:
	acTable();
	virtual ~acTable();

	/**
	 * @brief Initialize a table contains nRows and nColumns and position it on the page. 
	 * The default row height is set to the height of the table divided by the number of rows. 
	 * Similarly, the default column (or cell) width is set to the width of the table divided by the number of columns
	 * 
	 * @param nRows Number of rows
	 * @param nColumns Number of columns
	 * @param rcBoundRect Bounding rectangle
	 * @return int 1 if successful, 0 otherwise 
	 */
	int	InitTable(UINT nRows, UINT nColumns, const RECT& rcBoundRect);

	/**
	 * @brief Get the Cell object from the cell coordinates
	 * 
	 * @param nRow 0 indexed row of cell to return
	 * @param nCol 0 indexed column of cell to return
	 * @param bReturnMainCell if 1 return the main cell of a group of merged cells
	 * @return acReportCell* Cell object at specified row and column
	 */
	acReportCell* GetCell(UINT nRow, UINT nCol, int bReturnMainCell = 1);

	/**
	 * @brief Merge a group of cells into a single cell
	 * 
	 * @param nLeft 0 indexed column of first cell
	 * @param nTop 0 indexed row of first cell
	 * @param nRight 0 indexed column of last cell
	 * @param nBottom 0 indexed row of last cell
	 * @return int 1 if successful, 0 otherwise
	 */
	int	MergeCells(UINT nLeft, UINT nTop, UINT nRight, UINT nBottom);

	/**
	 * @brief Get the bounding rectangle for a group of cells
	 * 
	 * @param nStartX 0 indexed column of first cell
	 * @param nStartY 0 indexed row of first cell
	 * @param nEndX 0 indexed column of last cell
	 * @param nEndY 0 indexed row of last cell
	 * @param rc Bounding coordinates of specified cells relative to the origin of the table
	 */
	void GetCellsBoundRect(UINT nStartX, UINT nStartY, UINT nEndX, UINT nEndY, RECT& rc);

	/**
	 * @brief Set the width of a specific column
	 * 
	 * @param nCol 0 indexed column to modify
	 * @param nWidth new width of column
	 * @param bMoveNext if 1, the next column is moved to the left or right, if 0 the next column is enlarged or shrinked
	 * @return int 1 if successful, 0 otherwise
	 */
	int	SetColWidth(UINT nCol, UINT nWidth, int bMoveNext);

	/**
	 * @brief Set the height of a specific row
	 * 
	 * @param nRow 0 indexed row to modify
	 * @param nHeight new height of row
	 * @param bMoveNext if 1, the next row is moved up or down, if 0 the next row is enlarged or shrinked
	 * @return int 1 if successful, 0 otherwise
	 */
	int SetRowHeight(UINT nRow, UINT nHeight, int bMoveNext);


	/**
	 * @brief Draw the table on the page. To be called after the table is fully constructed
	 * 
	 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
	 */
	void Draw(HANDLE pdev);

private:
	UINT	rows;			// row count
	UINT	columns;		// col count
	RECT	boundRect;		// posistion and size

	UINT			rowCount;	// number of rows in this table
	acReportRow*	rowArray;	// row array
};

class acReportCell
{
public:
	typedef enum
	{
		MERGE_NONE = 0,
		MERGE_LEFT = 1,
		MERGE_RIGHT = 2,
		MERGE_TOP = 4,
		MERGE_BOTTOM = 8,
		MERGE_MAIN = 16,
	} MergeDirection;

	typedef enum
	{
		Left = 1,
		HCentered = 2,
		Right = 3,
	} HorzAlign;

	typedef enum
	{
		Top = 1,
		VCentered = 2,
		Bottom = 3,
	} VertAlign;

	typedef enum
	{
		acSolid = 0,
		acDashed = 1,
		acDotted = 2,
	} BorderStyles;

	typedef enum
	{
		acVertBorderNone = 0,
		acVertBorderLeft = 1,
		acVertBorderRight = 2,
		acVertBorderBoth = 3,
	} VertBorders;

	typedef enum
	{
		acHorzBorderNone = 0,
		acHorzBorderTop = 1,
		acHorzBorderBottom = 2,
		acHorzBorderBoth = 3,
	} HorzBorders;

	typedef	enum
	{
		acCellTitlePositionTopLeft = 1,
		acCellTitlePositionTopCenter = 2,
		acCellTitlePositionTopRight = 3,
		acCellTitlePositionBottomLeft = 4,
		acCellTitlePositionBottomCenter = 5,
		acCellTitlePositionBottomRight = 6
	} 	CellTitlePositionConstants;

	/**
	 * @brief Set the cell's main text content 
	 * 
	 * @param lpszText Pointer to a text string containing the new cell content. The lpszText pointer can be discarded after calling this method
	 */
	void SetText(LPCSTR lpszText);

	/**
	 * @brief Set the cell's title
	 * 
	 * @param lpszTitle Pointer to a text string containing the new cell title. The lpszTitle pointer can be discarded after calling this method
	 */
	void SetTitleText(LPCSTR lpszTitle);

public:
	/**
	 * Spacing between cell border and contents
	 */
	long			margin;

	/**
	 * Font used to render cell content. The font handle should be created with RTPdfGetFont
	 */
	HANDLE			font;
	/**
	 * Point size of font used to render contents
	 */
	float			pointSize;
	/**
	 * Image contained in cell. The backBmp handle should be created with RTPdfGetImage 
	 */
	HANDLE			backBmp;	// background bitmap
	/**
	 * Color used to fill the background of the cell, default value is transparent
	 * The backColor value is in RGB format
	 */
	COLORREF		backColor;
	/**
	 * Color used to render cell contents, default value is 0 or Black
	 * The textColor value is in RGB format
	 */
	COLORREF		textColor;
	/**
	 * Horizontal alignment of text or image contained in cell
	 */
	HorzAlign		horzAlign;
	/**
	 * Vertical alignment of text or image contained in cell
	 */
	VertAlign		vertAlign;
	/**
	 * Top and/or bottom cell borders
	 */
	HorzBorders		horzBorders;
	/**
	 * Left and/or right cell borders
	 */
	VertBorders		vertBorders;
	/**
	 * Width of cell borders
	 */
	int				borderWidth;
	/**
	 * Color of cell borders
	 * The borderColor value is in RGB format
	 */
	COLORREF		borderColor;
	/**
	 * Line style for cell borders
	 */
	BorderStyles	borderStyle;
	/**
	 * Specifies that text is to be drawn vertically
	 */
	int				vertical;
	/**
	 * If clipBoundaries is 1, cell contents are clipped to cell boundaries
	 */
	int				clipToBoundaries;
	/**
	 * Color of cell title
	 * The titleColor value is in RGB format
	 */
	COLORREF		titleColor;
	/**
	 * Font used to render cell title. The titleFont handle should be created with RTPdfGetFont
	 */
	HANDLE			titleFont;
	/**
	 * Point size of font used to render cell title
	 */
	float			titlePointSize;
	/**
	 * Position of cell title 
	 */
	CellTitlePositionConstants	titlePosition;	// title text position
									
private:
	acReportCell(acReportRow* parent);
	virtual ~acReportCell();

	UINT			x;			// horizonal position
	UINT			width;		// column (cell) width
	UINT			mainX;		// main cell in case of merged cells
	UINT			mainY;		// main cell in case of merged cells
	long			spanX;		// number of cells merged X direction
	long			spanY;		// number of cells merged Y direction
	MergeDirection	merge;		// type of merging for merged cells
	acReportRow		*row;

	LPTSTR			text;       // cell text content
	LPTSTR			titleText;	// title text

	// draw cell content
	void Draw(LPPDFINFO pPdfDC);
	// get bounding rect
	void GetBoundRect(RECT& rc);

	// return containing row
	acReportRow	*GetRow() { return row; }

	friend class acReportRow;
	friend class acTable;
};

class acReportRow
{
private:
	// modify cell width
	int	SetColWidth(UINT col, UINT width, int moveNext);

	// get number of columns
	UINT	GetColCount() { return colCount; }

	UINT		y;			// vertical position
	UINT		height;		// row height

	UINT			colCount;	// number of columns in this row
	acReportCell*	cells;		// cell array

	acTable		*GetTable() { return table; }
	acTable		*table;

	acReportRow(acTable *table);
	virtual ~acReportRow();

	// draw all cells to PDF device context
	void	DrawAllCells(LPPDFINFO pPdfDC);

	// initialize cell array
	int	InitCellArray(UINT nCellCount, UINT nRowWidth);

	friend class acTable;
	friend class acReportCell;

};
///@}

#endif // !defined(ACTABLE_H_INCLUDED_)
