//////////////////////////////////////////////////////////////////////////////////
//                                                                              //
// Amyuni RTPDF-32 - PDF Library for On Time RTOS-32                            //
//                                                                              //
// Copyright AMYUNI Lda. - AMYUNI Technologies                                  //
// 1998-2021, All Rights Reserved                   ***** CONFIDENTIAL *****    //
//                                                                              //
// Permission to use this work for any purpose must be expressly obtained       //
// from: AMYUNI Technologies, https://www.amyuni.com, management@amyuni.com     //
//                                                                              //
//////////////////////////////////////////////////////////////////////////////////
//
#pragma once

#ifdef __cplusplus
extern "C"
{
#endif

#ifndef RTPDF	// defined when called from within the library
typedef void*	LPPDFINFO;
typedef void*	HANDLE;
#endif

#include "RTPDF-DDI.h"

/** @name Library Initialization
* \anchor LibraryInit
*  Initialize the library, set license key, allocate and free internal structures
* @code
//
// This sample shows how to initialize and terminate the RTPDF-32 library
//
int main()
{
	LPPDFINFO	pdev = RTPdfInit();

	if (pdev)
	{
		// provide a valid license key depending on the version of the library and licensed features
		const char *szLicenseKey = "07EFCDAB...EA8B";
		RTPdfSetLicenseKey(pdev, "*****", szLicenseKey, strlen(szLicenseKey));

		// Add here code to render to PDF
		// ...

		// free all internal structures and terminate
		RTPdfEnd(pdev);
	}
}

* @endcode
*/
///@{

/**
 * @brief Allocate memory and initialize a PDFINFO structure
 * 
 * @return LPPDFINFO Pointer to allocated PDFINFO structure
 */
LPPDFINFO RTPdfInit();

/**
 * @brief Clear all memory allocated while creating a PDF file and free the PDFINFO structure
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit 
 */
void RTPdfEnd(LPPDFINFO pdev);

/**
 * @brief Set the License Key for the library. The license key contains specific licensed features and the library version number
 * Should be called directly after RTPdfInit
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @param szLicensee Name of licensee
 * @param szKey License key provided by Amyuni
 * @param nKeyLen Length in bytes of the szKey
 * @return int 0 if successful, a negative error code otherwise
 */
int RTPdfSetLicenseKey(LPPDFINFO pdev, const char *szLicensee, const char *szKey, int nKeyLen);

///@}

/** @name Document and Page Processing Functions
* \anchor DocumentProcessing
*  Start and End a PDF document, Start and End document pages
* @code
//
// This sample shows how to start a new PDF document and create a couple of pages
//
int main()
{
	LPPDFINFO	pdev = RTPdfInit();

	if (pdev)
	{
		// provide a valid license key depending on the version of the library and licensed features
		const char *szLicenseKey = "07EFCDAB...EA8B";
		RTPdfSetLicenseKey(pdev, "*****", szLicenseKey, strlen(szLicenseKey));

		// start a new PDF document
		RTPdfStartDoc(pdev);

		// start page 1
		RTPdfStartPage(pdev);

		// Add here code to render page 1
		// ...

		// end page 1
		RTPdfEndPage(pdev);

		// start page 2
		RTPdfStartPage(pdev);
		// Add here code to render page 2
		// ...

		// end page 2
		RTPdfEndPage(pdev);

		// signal end of document
		RTPdfEndDoc(pdev);

		// free all internal structures and terminate
		RTPdfEnd(pdev);
	}
}

* @endcode
*/
///@{

/**
 * @brief Start a new PDF document
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfStartDoc(LPPDFINFO pdev);

/**
 * @brief Start a new page, should be called after a call to RTPdfStartDoc and before a new page is started
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfStartPage(LPPDFINFO pdev);

/**
 * @brief End the current page
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfEndPage(LPPDFINFO pdev);

/**
 * @brief End the current document, no output is allowed after this call
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfEndDoc(LPPDFINFO pdev);

/**
* @brief Set the page dimensions and orientation of all pages following this call
*
* @param pdev Pointer to PDFINFO structure returned by RTPdfInit
* @param pdm Pointer to a Windows Devmode structure containing the page information. The following settings are currently supported:\n
*         dmOrientation - dmPaperSize - dmPaperLength - dmPaperWidth - dmFormName\n
* 
* @return int 1 if successful, 0 otherwise
* @par Sample Code
* @code
	// change the format of subsequent pages to Legal and the orientation to Landscape
	DEVMODEA	dm;
	memset(&dm, 0, sizeof(dm));
	dm.dmSize = sizeof(dm);
	lstrcpy((LPSTR)dm.dmFormName, "Legal");
	dm.dmOrientation = DMORIENT_LANDSCAPE;
	// specify which DEVMODE fields are being modified
	dm.dmFields |= (DM_FORMNAME | DM_ORIENTATION);
	RTPdfResetDevmode(pdev, &dm);

	// start new page
	RTPdfStartPage(pdev);
* @endcode
*/
int RTPdfResetDevmode(LPPDFINFO pdev, PDEVMODEA pdm);

/**
* @brief Shift all drawing operations by a certain distance
* To reset the origin, call the same function with opposite X and Y values
*
* @param pdev Pointer to PDFINFO structure returned by RTPdfInit
* @param lXOrg Shift along the X axis
* @param lYOrg Shift along the Y axis
*/
void RTPdfSetDrawingOrigin(LPPDFINFO pdev, long lXOrg, long lYOrg);

///@}

/** @name Document Metadata and Structure Functions
* \anchor DocumentStructure
*  
*/
///@{
/**
* @brief Enumeration of the various document info attributes that can be used in RTPdfSetDocumentInfo
*
*/
typedef enum _DocumentInfo
{
	DocumentInfoTitle = 0,
	DocumentInfoAuthor = 1,
	DocumentInfoCreator = 2,
	DocumentInfoSubject = 3,
	DocumentInfoKeywords = 4
} DocumentInfo;

/**
* @brief Set a document information attribute (also known as Metadata)\n
* The attributes should be set before calling RTPdfStartDoc and cannot be modified afterwards\n
* Document attributes in PDF use the PDFDocEncoding character encoding which is documented in the PDF specs\n
* Unicode strings can be set by prefixing the string with 0xFEFF. The macro INIT_UNICODE_STRING can be used to create unicode strings\n
*
* @param pdev Pointer to PDFINFO structure returned by RTPdfInit
* @param nDocumentInfo One of the DocumentInfo attributes to set
* @param szEntryValue Value of the attribute, either single-byte or douvle-byte
* @return int 1 if successful, 0 otherwise
* @par Sample Code
* @code
//
// set document info attributes in single-byte (PDFDocEncoding) or Unicode formats
//
UNICODE_STRING us;
INIT_UNICODE_STRING(us, "\x07\x68\x98\x98");	// simplified chinese for "Title"
RTPdfSetDocumentInfo(pdev, DocumentInfoTitle, (LPBYTE)&us);
RTPdfSetDocumentInfo(pdev, DocumentInfoAuthor, (LPBYTE)"Amyuni Technologies");
* @endcode
*/
int RTPdfSetDocumentInfo(LPPDFINFO pdev, DocumentInfo nDocumentInfo, const BYTE *szEntryValue);

/**
*
* @brief Specify a PDF layer on which to place the next objects
*
* @param pdev Pointer to PDFINFO structure returned by RTPdfInit
* @param pwstrLayerTitle In double-byte (Unicode) format. Can be one of three things:\n
* 1) If RTPdfSetLayer is called inside a page (between calls to StartPage and EndPage), pwstrLayerTitle is the title of the layer on which to place the objects\n
* 2) If RTPdfSetLayer is called outside a page, pwstrLayerTitle should contain the list and order in which to display the layer tree in a PDF viewer\n
* 3) If pwstrLayerTitle is an empty string, it closes the last opened layer and moves up one level\n
* @note The Order should be a string in the format "/Order[(Layer Title 1)(Layer Title 2)(...)]" for a single level tree of layers\n
*    Or "/Order[(Layer Title 1)[(Layer Subtitle 1)(Layer Subtitle 2)(...)] (Layer Title 2)[(Layer Subtitle 21)(Layer Subtitle 22)(...)]]" for a multi-level tree of layers
* @return int a negative value if an error occurs. When successful, returns the level of the layer in a multi-level tree of layers, this allows to properly close all the layers
* @par Sample Code 
* @code
	// set document information before starting the document
	SetDocumentInfoTest(pdev);

	RTPdfSetLayer(pdev, L"/Order[(Clipped Polygons)[(Clipped Polygons - 1)(Clipped Polygons - 2)](Some Text)]");

	if (!RTPdfStartDoc(pdev))
	{
		...
	}

	// output first page
	RTPdfStartPage(pdev);

	// place the next polygon in a PDF layer named "Clipped Polygons - 1" under "Clipped Polygons"
	RTPdfSetLayer(pdev, L"Clipped Polygons");
	RTPdfSetLayer(pdev, L"Clipped Polygons - 1");

	// output some polygons
	DrawPolygonTest(pdev);

	// end previous layers
	RTPdfSetLayer(pdev, L"");
	RTPdfSetLayer(pdev, L"");
* @endcode
*/
int RTPdfSetLayer(LPPDFINFO pdev, PWSTR pwstrLayerTitle);

/**
*
* @brief Create a hyperlink to an internal location (bookmark) or external location (URL)
*
* @param pdev Pointer to PDFINFO structure returned by RTPdfInit
* @param prclHyperLink Pointer to a rectangle containing the bounds of the hyperlink. If this parameter is NULL, the location of the last drawn object is used
* @param pwszDestination In double-byte (Unicode) format. Can be one of two things:\n
* 1) If pwszDestination starts with a number sign (#) the hyperlink points to an internal bookmark\n
* 2) If pwszDestination does not start with #, it should contain a valid external URL to which the hyperlink points\n
* @par Sample Code
* @code
// set a hyperlink on cell [1,1] to different location on the document (Bookmark)
table.GetCellsBoundRect(3, 3, 3, 3, rc);
// the cell coordinates are relative to the table origin and have to be offset by the origin of the table
RECTL rcl = { rc.left + rcTable.left, rc.top + rcTable.top, rc.right + rcTable.left, rc.bottom + rcTable.top };
RTPdfSetHyperLink(pdev, &rcl, L"#Table on Page 1");

RTPdfDrawImage(pdev, NULL, hImg4, FLOAT2LONG(36.0f), FLOAT2LONG(320.0f), FLOAT2LONG(150.0f), FLOAT2LONG(0.0f), FALSE);
// set a hyperlink on the last image object to an external URL
RTPdfSetHyperLink(pdev, NULL, L"https://www.amyuni.com");
* @endcode
*/
void RTPdfSetHyperLink(LPPDFINFO pdev, const PRECTL prclHyperLink, PWSTR pwszDestination);

/**
*
* @brief Add a bookmark to the current location in the bookmarks tree 
*
* @param pdev Pointer to PDFINFO structure returned by RTPdfInit
* @param nParent Parent ID of this bookmark. Set to 0 for the root bookmark or a value returned by a previous call to RTPdfSetBookmark
* @param pwstrBookmarkTitle Title of the bookmark in double-byte (Unicode) format
* @return int a negative value if an error occurs. When successful, returns the level of the bookmark which can be used as a parent to a subsequent bookmark
* @par Sample Code
* @code

// output first page
RTPdfStartPage(pdev);

// set a bookmark to the first table on the first page
RTPdfSetBookmark(pdev, 0, L"Table on Page 1");

// run the table creation test
DrawTableTest(pdev);

RTPdfEndPage(pdev);

// output second page
RTPdfStartPage(pdev);

// set a bookmark to the first table on the second page
RTPdfSetBookmark(pdev, 0, L"Table on Page 2");

// run the table creation test
DrawTableTest(pdev);
* @endcode
*/
int RTPdfSetBookmark(LPPDFINFO pdev, int nParent, PWSTR pwstrBookmarkTitle);

///@}

/** @name Callback Functions
 * \anchor Callbacks
 * These are callback functions from the library. The caller has to implement these functions in order to tell the library where to store the PDF data
 * The callback IsMetricCountry also configures the library for Letter or A4 papersize
 * @code
//
// This sample shows how to write the PDF file to a file under the c:\temp folder
//
HANDLE RTPdfOpenPort(const char *szDocTitle, const unsigned long dwJobId)
{
	return (HANDLE)CreateFile("c:\\temp\\test.pdf", GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
}

int RTPdfWritePort(HANDLE hPort, const unsigned char* pData, unsigned long *pdwDataSize)
{
	return WriteFile(hPort, pData, *pdwDataSize, pdwDataSize, NULL);
}

int RTPdfClosePort(HANDLE hPort, int bCancel)
{
	return CloseHandle(hPort);
}
* @endcode
*/
///@{
/**
* @brief Allows the caller to create a file or open a port to where the PDF data is sent
*
* @param szDocTitle Optional document title if set by RTPdfSetDocumentInfo
* @param dwJobId Reserved for future use
* @return HANDLE Pointer to the file or port opened by the caller
*/
HANDLE RTPdfOpenPort(const char *szDocTitle, const unsigned long dwJobId);

/**
* @brief Write a block of data to the specified file or port
*
* @param hPort Pointer to the file or port opened by the caller in RTPdfOpenPort
* @param pData Pointer to the data to be written
* @param pdwDataSize On input, number of bytes to write. On output, the caller should return the number of bytes actually written
* @return int 1 if successful, 0 otherwise
*/
int RTPdfWritePort(HANDLE hPort, const unsigned char* pData, unsigned long *pdwDataSize);

/**
* @brief Close the file or port that was opened in RTPdfOpenPort
*
* @param hPort Pointer to the file or port opened by the caller in RTPdfOpenPort
* @param bCancel 1 if the document was canceled, e.g. in the case of an error, 0 if the document terminated normally
* @return int 1 if successful, 0 otherwise
*/
int RTPdfClosePort(HANDLE hPort, int bCancel);

#ifndef RTPDF
/**
* @brief This function is used to set the default paper size to Letter or A4
* By default RTPDF-32 attempts to read the paper size from the system Locale settings and will only call IsMetricCountry if this is not successful
*
* @return int 1 to set the default paper size to A4 or 0 for Letter
*/
int IsMetricCountry();
#endif

///@}

/** @name Text Rendering Functions
* \anchor TextRendering
*  Font and Text handling functions
* @code
//
// This sample shows how to select a font and render some text on a page
//
	// document must be started before creating any fonts
	RTPdfStartDoc(pdev);

	// Load Helvetica (Arial) font
	HANDLE	hFnt1 = RTPdfGetFont(pdev, "Helvetica", 0, 0, 0, 1, 0);

	// start page
	RTPdfStartPage(pdev);
	
	// set font and text size
	RTPdfSetFont(pdev, hFnt1, 10.4f);
	// output some text
	RTPdfTextOut(pdev, NULL, NULL, FLOAT2LONG(100.0f), FLOAT2LONG(100.0f), 0.0f, (const BYTE *)"RTPDF Text Output", strlen("RTPDF Text Output"), 0);

	RTPdfEndPage(pdev);

* @endcode
*/
///@{

/**
 * @brief Create a PDF font object and return a handle that can be used to render text on a page
 * RTPdfStartDoc has to be called before any fonts are created
 * The PDF specifications define a set of 14 fonts that can be used within a PDF document without prior definition of these fonts
 * Because RTOS-32 has no support for fonts, only these 14 predefined fonts are upported by RTPDF-32
 * These fonts are: 
 *      Helvetica (equivalent to Arial), Helvetica-Bold, Helvetica-BoldItalic, Helvetica-Italic
 *      Times (equivalent to Times New Roman), Times-Bold, Times-BoldItalic, Times-Italic
 *      Courier (equivalent to Courier New), Courier-Bold, Courier-BoldItalic, Courier-Italic
 *      Symbol
 *      ITC Zapf Dingbats
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @param szFontName The font name and style to be used
 * @param bBold Reserved for future use
 * @param bItalic Reserved for future use
 * @param nCodePage Reserved for future use
 * @param bType1 Reserved for future use
 * @param bCID Reserved for future use
 * @return HANDLE to the font that can used in RTPdfSetFont, NULL of the function fails
 */
HANDLE RTPdfGetFont(LPPDFINFO pdev, const char *szFontName, int bBold, int bItalic, int nCodePage, int bType1, int bCID);

/**
 * @brief Set the font to be used in all text drawing operations after this function is called
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @param hFont Pointer to the font returned in RTPdfGetFont
 * @param fPointSize Text size in Points
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfSetFont(LPPDFINFO pdev, HANDLE hFont, float fPointSize);

/**
 * @brief Set the Word and Character spacing of any text drawn after this function is called
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @param lWordSpacing Spacing to be added to the space character (character code 32) 
 * @param lCharSpacing Spacing to be added between characters
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfSetTextSpacing(LPPDFINFO pdev, long lWordSpacing, long lCharSpacing);

/**
 * @brief 
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @param pco Pointer to a clipping object used to clip the path to a rectangular area
 * @param pboText Pointer to Brush object with which to paint the text
 * @param lPosX Horizontal position in PDF units
 * @param lPosY Vertical position in PDF units
 * @param fAngle Rotation angle of text in degrees and measured counter-clockwise
 * @param pbCharCodes Text encoded in WinAnsi character codes
 * @param nCharCount Number of characters in pbCharCodes
 * @param nEncoding Reserved for future use
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfTextOut(LPPDFINFO pdev, CLIPOBJ *pco, BRUSHOBJ *pboText, long lPosX, long lPosY, float fAngle, const BYTE *pbCharCodes, int nCharCount, int nEncoding);

/**
 * @brief Get width of a text string given the font name and size
 * 
 * @param hFont Pointer to the font returned in RTPdfGetFont
 * @param fPointSize Text size in Points 
 * @param pbCharCodes Text string
 * @param nCharCount Number of characters in text string
 * @param nEncoding Reserved for future use
 * @return float Width of text
 */
float RTPdfGetTextWidth(HANDLE hFont, float fPointSize, const BYTE *pbCharCodes, int nCharCount, int nEncoding);

/**
 * @brief Draw text given a bounding rectangle
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @param rcText Bounding rectangle
 * @param nTextAlign Combination of horizontal and vertical text alignement. Should be created using MAKELONG(horzAlign, vertAlign)
 * For values of horizontal and vertical alignment values check the acReportCell::HorzAlign and acReportCellVertAlign enumerators
 * @param bClipToBoundaries If 1 clip the text to cell rcText boundaries
 * @param crTextColor Color in GDI format to render the text, can be created using the RGB2GDICOLOR macro
 * @param pbCharCodes Text encoded in WinAnsi character codes
 * @param nCharCount Number of characters in pbCharCodes
 * @param nEncoding Reserved for future use
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfDrawText(LPPDFINFO pdev, RECT rcText, UINT nTextAlign, int bClipToBoundaries, COLORREF crTextColor, const BYTE *pbCharCodes, int nCharCount, int nEncoding);

///@}

/** @name Image Rendering Functions
* \anchor ImageRendering
*  
* @code
//
// This sample shows how to render an image
//
	// load an image from memory into the PDF object
	HANDLE	hImg1 = RTPdfGetImage(pdev, Test_Image_Jpeg, sizeof(Test_Image_Jpeg));

	// create a clipping path to crop the image
	CLIPOBJ	clipPath = {
		0,							// Region Identifier, should be 0
		{ FLOAT2LONG(36.0f), FLOAT2LONG(36.0f), FLOAT2LONG(80.0f), FLOAT2LONG(80.0f) },			// Rectangular region bounds	
		DC_RECT,					// Rectangular clipping region, the only supported value
		FC_RECT,					// Complexity of the region
		0,							// Not used
		0							// Not used
	};

	// draw image
	RTPdfDrawImage(pdev, &clipPath, hImg1, FLOAT2LONG(36.0f), FLOAT2LONG(36.0f), FLOAT2LONG(72.5f), FLOAT2LONG(0.0f));

* @endcode
*/
///@{

/**
 * @brief Output an image object to the PDF stream without yet drawing any image
 * BMP, JPeg and PNG images are currently supported. This call will fail if a specific image is not supported by the PDF specifications or by RTPDF-32
 * Example: PNG images with a bit depth different than 8 are not supported
 * The returned handle can later be used to draw the image on one or multiple pages
 * The RTPDF-32 library deletes the returned handle when the document is closed
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @param pbImageData Pointer to the image data in Jpeg, PNG or BMP formats
 * @param ulImageSize Size of the image data in pbImageData
 * @return HANDLE Pointer to the image object to be used in RTPdfDrawImage or NULL if the function fails
 */
HANDLE RTPdfGetImage(LPPDFINFO pdev, const BYTE *pbImageData, unsigned long ulImageSize);

/**
 * @brief Draw a preloaded image to a specific location on a page
 * All units should be provided in PDF units where 1 unit is 1/72 of an inch
 * If the destination height of the image is specified as 0.0f, the width of the image is adjusted to keep the original aspect ratio
 * If the destination width of the image is specified as 0.0f, the height of the image is adjusted to keep the original aspect ratio
 * If both width and height are specified and bKeepAspectRatio is 1, the image is centered in the rectangle while keeping its aspect ratio
 *
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @param pco Pointer to a clipping object used to clip the path
 * @param hImage Handle of an image returned by RTPdfGetImage
 * @param lStartX Left coordinate
 * @param lStartY Top coordinate
 * @param lWidth Destination width of the image
 * @param lHeight Destination height of the image
 * @param bKeepAspectRatio Flag to center the image while keeping its aspect ratio
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfDrawImage(LPPDFINFO pdev, CLIPOBJ *pco, HANDLE hImage, long lStartX, long lStartY, long lWidth, long lHeight, int bKeepAspectRatio);
///@}

/** @name Vector Rendering Functions
* \anchor VectorRendering
*  
* @code
//
// This sample shows how to draw a polygon with dashed lines and fill it with a solid color
//
	// create a path object to draw
    PATHOBJ	*ppo = RTPdfCreatePath(pdev);

	// start coordinates of our polygon
    POINTFIX pfx1 = { FLOAT2FIX(100.4f), FLOAT2FIX(100.0f) };
    // add 2 other vertices
	POINTFIX pfx2[] = { { FLOAT2FIX(200.6f), FLOAT2FIX(200.0f) },
						{ FLOAT2FIX(300.6f), FLOAT2FIX(200.0f) }
						};
    // define the stroked and fill brushes
	BRUSHOBJ stroke_brush = { RGB2GDIBRUSH(0x00, 0x55, 0xAF), 0, 0 };	// blue-ish
	BRUSHOBJ fill_brush = { RGB2GDIBRUSH(0x00, 0xAF, 0x55), 0, 0 };		// green-ish

	// create a dashed line with a proportion of 5 to 1
	FLOAT_LONG	DashedLineStyle[2];
	DashedLineStyle[0].l = 10;
	DashedLineStyle[1].l = 2;
	LINEATTRS	DashedLine = {
		0,							// Option flags, should be 0
		JOIN_ROUND,					// Join style
		ENDCAP_ROUND,				// End cap style
		0,							// Line width
		0,							// Not used
		_countof(DashedLineStyle),	// Number of entries in the style array
		DashedLineStyle,			// Style array for non-continuous lines
		0							// Not used
	};
	DashedLine.elWidth.l = 10;		// line width in pixels x 10

	PATHOBJ_bMoveTo(ppo, pfx1);
	PATHOBJ_bPolyLineTo(ppo, pfx2, _countof(pfx2));
	// close our figure
	PATHOBJ_bPolyLineTo(ppo, &pfx1, 1);

	RTPdfStrokeAndFillPath(pdev, ppo, NULL, &stroke_brush, &DashedLine, &fill_brush, NULL, 0, PDF_PATH_FILLANDSTROKE);

    // always delete the path object when done
	RTPdfDeletePath(ppo);

* @endcode
*/
///@{

/**
* @brief Create a path object made of lines and curves to be used in all vector drawing operations\n 
* See \ref PATHOBJ_bMoveTo, \ref PATHOBJ_bPolyLineTo and \ref PATHOBJ_bPolyBezierTo for how to add elements to the path
*
* @param pdev Pointer to PDFINFO structure returned by RTPdfInit
* @return PATHOBJ* Pointer to path object
*/
PATHOBJ * RTPdfCreatePath(LPPDFINFO pdev);

/**
* @brief Delete the path object created through RTPdfCreatePath
*
* @param ppo Path object returned by RTPdfCreatePath
* @return void
*/
void RTPdfDeletePath(PATHOBJ *ppo);

// fill and stoke values available for RTPdfStrokeAndFillPath::flOptions
#define	PDF_PATH_NONE			0
#define	PDF_PATH_STROKE			1
#define PDF_PATH_FILL			2
#define PDF_PATH_FILLANDSTROKE	3
#define PDF_PATH_FILL_WINDING	8

/**
 * @brief Draw a path object made of straight lines and/or Bezier curves
 * If requested, also fill the path using the specified brush
 * 
 * @param pdev Pointer to PDFINFO structure returned by RTPdfInit
 * @param ppo Pointer to a path object returned by RTPdfCreatePath
 * @param pco Pointer to a clipping object used to clip the path to a rectangular area
 * @param pboStroke Brush object used to draw the outline of the path
 * @param plineattrs Line attributes that define things such as width of the line, line ending style and so on
 * @param pboFill Brush object used to fill the path
 * @param pptlBrushOrg Reserved for future use
 * @param mix  Reserved for future use
 * @param flOptions A combination of PDF_PATH options as defined above
 * @return int 1 if successful, 0 otherwise
 */
int RTPdfStrokeAndFillPath(LPPDFINFO pdev, PATHOBJ *ppo, CLIPOBJ *pco, 
							BRUSHOBJ *pboStroke, PLINEATTRS plineattrs,
							BRUSHOBJ *pboFill, PPOINTL pptlBrushOrg, ULONG mix, FLONG flOptions);

///@}

#ifdef __cplusplus
}
#endif
