//////////////////////////////////////////////////////////////////////////////////
//                                                                              //
// Amyuni RTPDF-32 - PDF Library for On Time RTOS-32                            //
//                                                                              //
// Copyright AMYUNI Lda. - AMYUNI Technologies                                  //
// 1998-2021, All Rights Reserved                   ***** CONFIDENTIAL *****    //
//                                                                              //
// Permission to use this work for any purpose must be expressly obtained       //
// from: AMYUNI Technologies, https://www.amyuni.com, management@amyuni.com     //
//                                                                              //
//////////////////////////////////////////////////////////////////////////////////
//
// RTPDF-Test.cpp : This file contains the 'main' function.
//

#include "pch.h"
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

// forward declarations
void SetDocumentInfoTest(LPPDFINFO pdev);
void DrawPolygonTest(LPPDFINFO pdev);
void DrawTextTest(LPPDFINFO pdev);
void DrawImageTest(LPPDFINFO pdev);
void DrawTableTest(LPPDFINFO pdev);

int main()
{
	// Intialize random number generator, this is needed to generate a pseudo-unique ID for a PDF document
	srand((UINT)time(NULL));

	LPPDFINFO	pdev = RTPdfInit();

	if (pdev)
	{
		/// replace by your actual licensee name and license key
		const char *szLicensee = "Licensee";
		const char *szLicenseKey = "07EFCD...55B63E";
		///
		if (RTPdfSetLicenseKey(pdev, szLicenseKey, szLicenseKey, strlen(szLicenseKey)) < 0)
		{
			// something is wrong with the license, a demo watermark will be printed across pages
			printf("Wrong RTPDF License Key, a demo watermark will be printed across pages\n");
		}

		// set document information before starting the document
		SetDocumentInfoTest(pdev);

		// set the names and order of all layers, this can be called anywhere outside of a page, preferrably before StartDoc
		RTPdfSetLayer(pdev, L"/Order[(Clipped Polygons)[(Clipped Polygons - 1)(Clipped Polygons - 2)] (Some Text)]");

		// document must be started before creating any fonts or images
		if (!RTPdfStartDoc(pdev))
		{
			// failed starting the document, probably an error in RTPdfOpenPort
			printf("Failed starting a new document, check check that RTPdfOpenPort is successful\n");

			// rendering of the PDF can continue but no output will be generated
		}
		
		// start first page
		RTPdfStartPage(pdev);

		// set a bookmark to the first table on the first page
		RTPdfSetBookmark(pdev, 0, L"Table on Page 1");

		// run the table creation test
		DrawTableTest(pdev);

		// place the next polygon in a PDF layer named "Clipped Polygons - 1" under "Clipped Polygons"
		RTPdfSetLayer(pdev, L"Clipped Polygons");
		RTPdfSetLayer(pdev, L"Clipped Polygons - 1");

		// output some polygons
		DrawPolygonTest(pdev);

		// end previous layers
		RTPdfSetLayer(pdev, L"");
		RTPdfSetLayer(pdev, L"");

		// end first page
		RTPdfEndPage(pdev);

		// change the format of subsequent pages to Legal and the orientation to Landscape
		DEVMODEA	dm;
		memset(&dm, 0, sizeof(dm));
		dm.dmSize = sizeof(dm);
		lstrcpy((LPSTR)dm.dmFormName, "Legal");
		dm.dmOrientation = DMORIENT_LANDSCAPE;
		// specify which DEVMODE fields are being modified
		dm.dmFields |= (DM_FORMNAME | DM_ORIENTATION);
		RTPdfResetDevmode(pdev, &dm);

		// start second page
		RTPdfStartPage(pdev);
		
		// output some images
		DrawImageTest(pdev);

		// place the next text in a PDF layer named "Some Text""
		RTPdfSetLayer(pdev, L"Some Text");

		// output some text
		DrawTextTest(pdev);

		// end previous layer
		RTPdfSetLayer(pdev, L"");

		// set a bookmark to the first table on the second page
		RTPdfSetBookmark(pdev, 0, L"Table on Page 2");

		// run the table creation test
		//DrawTableTest(pdev);

		// place the next polygon in a PDF layer named "Clipped Polygons - 2" under "Clipped Polygons"
		RTPdfSetLayer(pdev, L"Clipped Polygons");
		RTPdfSetLayer(pdev, L"Clipped Polygons - 2");

		// output some polygons
		DrawPolygonTest(pdev);

		// end previous layer
		RTPdfSetLayer(pdev, L"");
		RTPdfSetLayer(pdev, L"");

		// end second page
		RTPdfEndPage(pdev);
		
		// end document
		RTPdfEndDoc(pdev);

		// free pdev and other library objects
		RTPdfEnd(pdev);
	}
}

void SetDocumentInfoTest(LPPDFINFO pdev)
{
//
// Set various document attributes in single or double character formats
//
	UNICODE_STRING us;

	RTPdfSetDocumentInfo(pdev, DocumentInfoTitle, (LPBYTE)"My Document");
	RTPdfSetDocumentInfo(pdev, DocumentInfoAuthor, (LPBYTE)"Amyuni Technologies");
	RTPdfSetDocumentInfo(pdev, DocumentInfoCreator, (LPBYTE)"RTPDF-32 v1.0");

	INIT_UNICODE_STRING(us, "\x2A\x59\x1F\x61\x22\x8C\x86\x4E");	// simplified chinese for Thank you so much
	RTPdfSetDocumentInfo(pdev, DocumentInfoSubject, (LPBYTE)&us);

	RTPdfSetDocumentInfo(pdev, DocumentInfoKeywords, (LPBYTE)"RTPDF,Test");
}

HANDLE GetImageFromFile(LPPDFINFO pdev, LPCSTR lpszFileName)
{
	//
	// Open image file from disk and load it into the PDF object
	//
	DWORD dwFileSize, dwFileSizeHigh;
	HANDLE hImg = NULL;
	HANDLE hf = CreateFile(lpszFileName, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hf != INVALID_HANDLE_VALUE)
	{
		dwFileSize = GetFileSize(hf, &dwFileSizeHigh);
		LPBYTE lpb = (LPBYTE)malloc(dwFileSize);
		if (lpb)
		{
			// read the image data into memory
			ReadFile(hf, lpb, dwFileSize, &dwFileSizeHigh, NULL);

			// save the image to the PDF object
			hImg = RTPdfGetImage(pdev, lpb, dwFileSize);

			// image data is already output to the PDF, can clear the data now
			free(lpb);
		}
		CloseHandle(hf);
	}

	return hImg;
}

void DrawTextTest(LPPDFINFO pdev)
{
	// Load fonts
	HANDLE	hFnt1 = RTPdfGetFont(pdev, "Helvetica", 0, 0, 0, 1, 0);
	HANDLE	hFnt2 = RTPdfGetFont(pdev, "Helvetica-Bold", 0, 0, 0, 1, 0);

	// output some text
	const char TEXT1[] = "RTPDF Text Output 1";
	const char TEXT2[] = "RTPDF Text Output 2";
	const char TEXT3[] = "RTPDF Rotated Text Output 1";
	const char TEXT4[] = "RTPDF Rotated Text Output 2";

	BRUSHOBJ	brush = { RGB2GDIBRUSH(0x00, 0x55, 0xAF), 0, 0 };	// blue-ish
	BRUSHOBJ	brush2 = { RGB2GDIBRUSH(0xAF, 0x00, 0x55), 0, 0 };	// red-ish

	RTPdfSetFont(pdev, hFnt1, 10.4f);
	RTPdfSetTextSpacing(pdev, FLOAT2LONG(2.0f), FLOAT2LONG(1.0f));
	RTPdfTextOut(pdev, NULL, &brush, FLOAT2LONG(36.5f), FLOAT2LONG(80.0f), 0.0f, (const BYTE *)TEXT1, strlen(TEXT1), 0);

	RTPdfSetTextSpacing(pdev, 0, 0);
	RTPdfTextOut(pdev, NULL, &brush, FLOAT2LONG(36.5f), FLOAT2LONG(90.0f), 0.0f, (const BYTE *)TEXT1, strlen(TEXT1), 0);
	RTPdfTextOut(pdev, NULL, &brush, FLOAT2LONG(36.5f), FLOAT2LONG(200.0f), 45.0f, (const BYTE *)TEXT3, strlen(TEXT3), 0);

	// create a clipping path to crop the text
	CLIPOBJ	clipPath = {
		0,							// Region Identifier, should be 0
		{ FLOAT2LONG(36.0f), FLOAT2LONG(136.0f), FLOAT2LONG(80.0f), FLOAT2LONG(300.0f) },			// Rectangular region bounds	
		DC_RECT,					// Rectangular clipping region, the only supported value
		FC_RECT,					// Complexity of the region
		0,							// Not used
		0							// Not used
	};

	// output some text with a bold font and clipping
	RTPdfSetFont(pdev, hFnt2, 10.4f);
	RTPdfTextOut(pdev, &clipPath, &brush2, FLOAT2LONG(36.5f), FLOAT2LONG(210.0f), 0.0f, (const BYTE *)TEXT2, strlen(TEXT2), 0);
	RTPdfTextOut(pdev, &clipPath, &brush2, FLOAT2LONG(36.5f), FLOAT2LONG(300.0f), 45.0f, (const BYTE *)TEXT4, strlen(TEXT4), 0);
}

void DrawImageTest(LPPDFINFO pdev)
{
	// Load images
	HANDLE  hImg1 = GetImageFromFile(pdev, "c:\\temp\\test.png");
	HANDLE  hImg2 = GetImageFromFile(pdev, "c:\\temp\\logo.jpg");
	HANDLE  hImg3 = GetImageFromFile(pdev, "c:\\temp\\logo-256.bmp");
	// load an image from memory into the PDF object
	HANDLE	hImg4 = RTPdfGetImage(pdev, Test_Image_Jpeg, sizeof(Test_Image_Jpeg));

	// create a clipping path to crop the image
	CLIPOBJ	clipPath = {
		0,							// Region Identifier, should be 0
		{ FLOAT2LONG(36.0f), FLOAT2LONG(36.0f), FLOAT2LONG(80.0f), FLOAT2LONG(80.0f) },			// Rectangular region bounds	
		DC_RECT,					// Rectangular clipping region, the only supported value
		FC_RECT,					// Complexity of the region
		0,							// Not used
		0							// Not used
	};

	// draw images
	RTPdfDrawImage(pdev, &clipPath, hImg1, FLOAT2LONG(36.0f), FLOAT2LONG(36.0f), FLOAT2LONG(72.5f), FLOAT2LONG(0.0f), FALSE);
	RTPdfDrawImage(pdev, NULL, hImg2, FLOAT2LONG(36.0f), FLOAT2LONG(120.0f), FLOAT2LONG(150.0f), FLOAT2LONG(0.0f), FALSE);
	RTPdfDrawImage(pdev, NULL, hImg3, FLOAT2LONG(36.0f), FLOAT2LONG(220.0f), FLOAT2LONG(150.0f), FLOAT2LONG(0.0f), FALSE);
	RTPdfDrawImage(pdev, NULL, hImg4, FLOAT2LONG(36.0f), FLOAT2LONG(320.0f), FLOAT2LONG(150.0f), FLOAT2LONG(0.0f), FALSE);

	// set a hyperlink on the last image object to an external URL
	RTPdfSetHyperLink(pdev, NULL, L"https://www.amyuni.com");
}


void DrawPolygonTest(LPPDFINFO pdev)
{
	PATHOBJ	*ppo = RTPdfCreatePath(pdev);
	// Path coordinates as defined by Windows GDI are in fixed point units, which is the value multipled by 16 for higher precision
	POINTFIX pfx1 = { FLOAT2FIX(100.4f), FLOAT2FIX(100.0f) };
	POINTFIX pfx2[] = { { FLOAT2FIX(200.6f), FLOAT2FIX(200.0f) },
						{ FLOAT2FIX(300.6f), FLOAT2FIX(200.0f) }
						};
	POINTFIX pfx3[] = { { FLOAT2FIX(300.6f), FLOAT2FIX(220.0f) },
						{ FLOAT2FIX(450.6f), FLOAT2FIX(230.0f) },
						{ FLOAT2FIX(300.6f), FLOAT2FIX(180.0f) }
						};
	BRUSHOBJ stroke_brush = { RGB2GDIBRUSH(0x00, 0x55, 0xAF), 0, 0 };	// blue-ish
	BRUSHOBJ fill_brush = { RGB2GDIBRUSH(0x00, 0xAF, 0x55), 0, 0 };		// green-ish

	LINEATTRS	ContinuousThickLine = {
		0,					// Option flags, should be 0
		JOIN_ROUND,			// Join style
		ENDCAP_ROUND,		// End cap style
		0,					// Line width
		0,					// Not used
		0,					// Number of entries in the style array
		NULL,				// Style array for non-continuous lines
		0					// Not used
	};
	ContinuousThickLine.elWidth.l = 20;		// line width in pixels x 10

	// create a dashed line with a proportion of 5 to 1
	FLOAT_LONG	DashedLineStyle[2];
	DashedLineStyle[0].l = 10;
	DashedLineStyle[1].l = 2;
	LINEATTRS	DashedLine = {
		0,							// Option flags, should be 0
		JOIN_ROUND,					// Join style
		ENDCAP_ROUND,				// End cap style
		0,							// Line width
		0,							// Not used
		_countof(DashedLineStyle),	// Number of entries in the style array
		DashedLineStyle,			// Style array for non-continuous lines
		0							// Not used
	};
	DashedLine.elWidth.l = 10;		// line width in pixels x 10

	CLIPOBJ	clipPath = {
		0,							// Region Identifier, should be 0
		{ FLOAT2LONG(200.0f), FLOAT2LONG(150.0f), FLOAT2LONG(400.0f), FLOAT2LONG(350.0f) },			// Rectangular region bounds	
		DC_RECT,					// Rectangular clipping region, the only supported value
		FC_RECT,					// Complexity of the region
		0,							// Not used
		0							// Not used
	};

	PATHOBJ_bMoveTo(ppo, pfx1);
	PATHOBJ_bPolyLineTo(ppo, pfx2, _countof(pfx2));
	PATHOBJ_bPolyBezierTo(ppo, pfx3, _countof(pfx3));
	// close our random figure
	PATHOBJ_bPolyLineTo(ppo, &pfx1, 1);

	RTPdfStrokeAndFillPath(pdev, ppo, &clipPath, &stroke_brush, &DashedLine, &fill_brush, NULL, 0, PDF_PATH_FILLANDSTROKE);

	// set a hyperlink on the last path object to an external URL
	RTPdfSetHyperLink(pdev, NULL, L"https://www.amyuni.com");

	RTPdfDeletePath(ppo);
}

void DrawTable(LPPDFINFO pdev, float left, float top)
{
	acTable	table;
	RECT	rc, rcTable = { FLOAT2LONG(left), FLOAT2LONG(top), FLOAT2LONG(left+530.0f), FLOAT2LONG(top+240.0f) };
	int		rows = 5, cols = 4;
	acReportCell *cell;

	table.InitTable(rows, cols, rcTable);

	table.SetColWidth(0, FLOAT2LONG(150.0f), TRUE);		// enlarge first column
	table.SetColWidth(1, FLOAT2LONG(75.0f), TRUE);		// reduce second column
	table.SetColWidth(2, FLOAT2LONG(175.0f), FALSE);	// enlarge column 3 and reduce column 4
	
	table.SetRowHeight(0, FLOAT2LONG(25.0f), 1);		// header row a bit smaller

	// set a default font for all the table
	HANDLE	hFnt1 = RTPdfGetFont(pdev, "Helvetica", 0, 0, 0, 1, 0);
	HANDLE  hFnt2 = RTPdfGetFont(pdev, "Helvetica-Bold", 0, 0, 0, 1, 0);

	for (int r = 0; r < rows; r++)
		for (int c = 0; c < cols; c++)
		{
			acReportCell *cell = table.GetCell(r, c);

			cell->horzBorders = acReportCell::HorzBorders::acHorzBorderBoth;
			cell->font = hFnt1;
			cell->pointSize = 11.5f;
			cell->titleFont = hFnt2;
			cell->titlePointSize = 9.0f;
			cell->borderWidth = 10;

			// make the first row different from the others
			if (r == 0)
			{
				cell->backColor = RGB(0, 0xAA, 0xFF);
				cell->textColor = RGB(0xFF, 0xFF, 0xFF);
			}
			else
			{
				cell->vertBorders = acReportCell::VertBorders::acVertBorderBoth;

				if (c == cols - 1)
				{
					// text in last column is right-aligned
					cell->horzAlign = acReportCell::HorzAlign::Right;
				}
			}
		}

	// merge the cells of the first row and set some text
	table.MergeCells(0, 0, cols - 1, 0);
	cell = table.GetCell(0, 0);
	cell->horzAlign = acReportCell::HorzAlign::HCentered;
	cell->SetText("Header Row");

	// merge 4 cells and set a background image and a title
	table.MergeCells(1, 2, 2, 3);
	cell = table.GetCell(2, 1);
	cell->horzAlign = acReportCell::HorzAlign::HCentered;
	cell->backBmp = GetImageFromFile(pdev, "c:\\temp\\test.png");
	cell->SetTitleText("Image with a title");

	cell = table.GetCell(3, 0);
	cell->SetTitleText("Title");
	cell->SetText("Cell contents clipped to boundaries");
	cell->clipToBoundaries = TRUE;

	cell = table.GetCell(2, cols - 1);
	cell->SetText("0.00");
	
	cell = table.GetCell(3, cols - 1);
	cell->SetText("100.00");
	cell->textColor = RGB(0, 0, 0xFF);

	table.Draw(pdev);

	// set a hyperlink on cell [1,1] to different location on the document (Bookmark)
	table.GetCellsBoundRect(3, 3, 3, 3, rc);
	// the cell coordinates are relative to the table origin and have to be offset by the origin of the table
	RECTL rcl = { rc.left + rcTable.left, rc.top + rcTable.top, rc.right + rcTable.left, rc.bottom + rcTable.top };
	RTPdfSetHyperLink(pdev, &rcl, L"#Table on Page 1");
}

void DrawTableTest(LPPDFINFO pdev)
{
	// draw same table twice for testing
	DrawTable(pdev, 50.0f, 50.0f);
	DrawTable(pdev, 50.0f, 400.0f);
}

int IsMetricCountry()
{
	return 0;	// default to Letter paper size
}

HANDLE RTPdfOpenPort(const char *szDocTitle, const unsigned long dwJobId)
{
	// construct a file name in the temp folder based on the document title
	CHAR fileName[MAX_PATH];
	sprintf_s(fileName, sizeof(fileName), "c:\\temp\\%s.pdf", (szDocTitle && szDocTitle[0] && szDocTitle[0] != -1) ? szDocTitle : "NoTitle");	// if we have a valid document title, use it, otherwise create "NoTitle.pdf"
																															// szDocTitle[0] is -1 (0xFF) when the title is Unicode

	HANDLE hFile = (HANDLE)CreateFile(fileName, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hFile == INVALID_HANDLE_VALUE)
	{
		printf("Error opening file %s for writing, error code %d\n", fileName, GetLastError());
		return NULL;
	}
	else
	{
		return hFile;
	}
}

int RTPdfWritePort(HANDLE hPort, const unsigned char* pData, unsigned long *pdwDataSize)
{
	if (!WriteFile(hPort, pData, *pdwDataSize, pdwDataSize, NULL))
	{
		printf("Error writing to file, error code %d\n", GetLastError());
		return FALSE;
	}
	else
	{
		return TRUE;
	}
}

int RTPdfClosePort(HANDLE hPort, int bCancel)
{
	return CloseHandle(hPort);
}


